<?php
/*
 * Elementor Varaus Case Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Varaus_Case extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'wpo-varaus_room';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Room Filter', 'varaus-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-posts-group';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Varaus Case widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['wpo_varaus_room_filter'];
	}

	/**
	 * Register Varaus Case widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{


		$posts = get_posts('post_type="project"&numberposts=-1');
		$PostID = array();
		if ($posts) {
			foreach ($posts as $post) {
				$PostID[$post->ID] = $post->ID;
			}
		} else {
			$PostID[__('No ID\'s found', 'varaus')] = 0;
		}


		$this->start_controls_section(
			'section_room',
			[
				'label' => esc_html__('Case Options', 'varaus-core'),
			]
		);
		$this->add_control(
			'room_filter',
			[
				'label' => esc_html__('Case Filter', 'varaus-core'),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Show', 'varaus-core'),
				'label_off' => esc_html__('Hide', 'varaus-core'),
				'return_value' => 'true',
				'default' => 'true',
			]
		);
		$this->end_controls_section(); // end: Section


		$this->start_controls_section(
			'section_room_listing',
			[
				'label' => esc_html__('Listing Options', 'varaus-core'),
			]
		);
		$this->add_control(
			'room_limit',
			[
				'label' => esc_html__('Case Limit', 'varaus-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 100,
				'step' => 1,
				'default' => 3,
				'description' => esc_html__('Enter the number of items to show.', 'varaus-core'),
			]
		);
		$this->add_control(
			'room_order',
			[
				'label' => __('Order', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'ASC' => esc_html__('Asending', 'varaus-core'),
					'DESC' => esc_html__('Desending', 'varaus-core'),
				],
				'default' => 'DESC',
			]
		);
		$this->add_control(
			'room_orderby',
			[
				'label' => __('Order By', 'varaus-core'),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'none' => esc_html__('None', 'varaus-core'),
					'ID' => esc_html__('ID', 'varaus-core'),
					'author' => esc_html__('Author', 'varaus-core'),
					'title' => esc_html__('Title', 'varaus-core'),
					'date' => esc_html__('Date', 'varaus-core'),
				],
				'default' => 'date',
			]
		);
		$this->add_control(
			'room_show_category',
			[
				'label' => __('Certain Categories?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => Controls_Helper_Output::get_terms_names('project_category'),
				'multiple' => true,
			]
		);
		$this->add_control(
			'room_show_id',
			[
				'label' => __('Certain ID\'s?', 'varaus-core'),
				'type' => Controls_Manager::SELECT2,
				'default' => [],
				'options' => $PostID,
				'multiple' => true,
			]
		);
		$this->add_control(
			'short_content',
			[
				'label' => esc_html__('Excerpt Length', 'varaus-core'),
				'type' => Controls_Manager::NUMBER,
				'min' => 1,
				'step' => 1,
				'default' => 16,
				'description' => esc_html__('How many words you want in short content paragraph.', 'varaus-core'),
				'condition' => [
					'room_style' => array('style-one'),
				],
			]
		);
		$this->add_control(
			'all_room',
			[
				'label' => esc_html__('All room Text', 'varaus-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__('Type your all room text here', 'varaus-core'),
			]
		);
		$this->add_control(
			'read_more_txt',
			[
				'label' => esc_html__('Check Availability', 'varaus-core'),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__('Type your Read More text here', 'varaus-core'),
			]
		);
		$this->end_controls_section(); // end: Section


		// Background
		$this->start_controls_section(
			'section_background_style',
			[
				'label' => esc_html__('Background', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'section_bg_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-studies-section .room-text-hide' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_title_typography',
				'selector' => '{{WRAPPER}} .room-studies-section .room-text-hide h2',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-studies-section .room-text-hide h2 ' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .room-studies-section .room-text-hide h2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Subtitle
		$this->start_controls_section(
			'section_subtitle_style',
			[
				'label' => esc_html__('Subtitle', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_subtitle_typography',
				'selector' => '{{WRAPPER}} .room-text-hide span',
			]
		);
		$this->add_control(
			'subtitle_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-text-hide span' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section

		// Details
		$this->start_controls_section(
			'section_content_style',
			[
				'label' => esc_html__('Content', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_content_typography',
				'selector' => '{{WRAPPER}} .room-text-hide p',
			]
		);
		$this->add_control(
			'content_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-text-hide p' => 'color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section


		// Button
		$this->start_controls_section(
			'section_nav_style',
			[
				'label' => esc_html__('Button', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'nav_color',
			[
				'label' => esc_html__('Button Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-text-hide .theme-btn-s2' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'nav_bg_color',
			[
				'label' => esc_html__('BG Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .room-text-hide .theme-btn-s2' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section



	}

	/**
	 * Render Case widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$room_style = !empty($settings['room_style']) ? $settings['room_style'] : '';
		$room_column = !empty($settings['room_column']) ? $settings['room_column'] : '';

		$room_limit = !empty($settings['room_limit']) ? $settings['room_limit'] : '';
		$room_order = !empty($settings['room_order']) ? $settings['room_order'] : '';
		$room_orderby = !empty($settings['room_orderby']) ? $settings['room_orderby'] : '';
		$room_show_category = !empty($settings['room_show_category']) ? $settings['room_show_category'] : [];
		$room_show_id = !empty($settings['room_show_id']) ? $settings['room_show_id'] : [];
		$all_room = !empty($settings['all_room']) ? $settings['all_room'] : '';

		$room_filter  = (isset($settings['room_filter']) && ('true' == $settings['room_filter'])) ? true : false;

		$all_room = $all_room ? $all_room : esc_html__('All', 'varaus-core');

		$read_more_txt = !empty($settings['read_more_txt']) ? $settings['read_more_txt'] : '';
		$read_more_txt = $read_more_txt ? $read_more_txt : esc_html__('Check Availability', 'varaus-core');

		// Turn output buffer on
		ob_start();

		// Pagination
		global $paged;
		if (get_query_var('paged'))
			$my_page = get_query_var('paged');
		else {
			if (get_query_var('page'))
				$my_page = get_query_var('page');
			else
				$my_page = 1;
			set_query_var('paged', $my_page);
			$paged = $my_page;
		}

		if ($room_show_id) {
			$room_show_id = json_encode($room_show_id);
			$room_show_id = str_replace(array('[', ']'), '', $room_show_id);
			$room_show_id = str_replace(array('"', '"'), '', $room_show_id);
			$room_show_id = explode(',', $room_show_id);
		} else {
			$room_show_id = '';
		}

		$args = array(
			// other query params here,
			'paged' => $my_page,
			'post_type' => 'hb_room',
			'posts_per_page' => (int)$room_limit,
			'category_name' => implode(',', $room_show_category),
			'orderby' => $room_orderby,
			'order' => $room_order,
			'post__in' => $room_show_id,
		);

		$varaus_room = new \WP_Query($args);

		if ($varaus_room->have_posts()) :
			$terms = get_terms('hb_room_type'); ?>
			<div class="room-studies-section">
				<div class="room-container sortable-gallery">
					<div class="row">
						<div class="col col-xs-12">
							<div class="gallery-filters">
								<?php if ($room_filter) { ?>
									<ul>
										<li><a data-filter="*" href="#" class="current"><?php echo esc_html($all_room) ?></a></li>
										<?php foreach ($terms as $term) :  ?>
											<li><a data-filter=".<?php echo esc_attr($term->slug); ?>" href="#"><?php echo esc_html($term->name); ?></a></li>
										<?php endforeach; ?>
									</ul>
								<?php } ?>
							</div>
							<div class="gallery-container room-grids">
								<?php
								while ($varaus_room->have_posts()) : $varaus_room->the_post();

									$room_options = get_post_meta(get_the_ID(), 'room_options', true);
									$room_image = isset($room_options['room_image']) ? $room_options['room_image'] : '';
									$starting_price = isset($room_options['starting_price']) ? $room_options['starting_price'] : '';

									global $post;
									$image_url = wp_get_attachment_url($room_image);
									$image_alt = get_post_meta($room_image, '_wp_attachment_image_alt', true);

									$terms = wp_get_post_terms(get_the_ID(), 'hb_room_type');

									foreach ($terms as $term) {
										$cat_class = $term->slug;
									}
									$count = count($terms);
									$i = 0;
									$cat_class = '';
									if ($count > 0) {
										foreach ($terms as $term) {
											$i++;
											$cat_class .= $term->slug . ' ';
											if ($count != $i) {
												$cat_class .= '';
											} else {
												$cat_class .= '';
											}
										}
									}

								?>
									<div class="grid <?php echo esc_attr($cat_class); ?>">
										<div class="room-item">
											<div class="img-holder">
												<?php if ($image_url) {
													echo '<img src="' . esc_url($image_url) . '" alt="' . esc_attr($image_alt) . '">';
												} ?>
											</div>
											<div class="room-text-show">
												<h2><?php echo get_the_title(); ?></h2>
											</div>
											<div class="room-text-hide">
												<h2><?php echo get_the_title(); ?></h2>
												<span><?php echo esc_html($term->name); ?></span>
												<p><?php echo wp_trim_words(get_the_content(), 20, ' '); ?></p>
												<small><?php echo esc_html__('From: ', 'varaus'); ?><span><?php echo esc_html($starting_price); ?></span><?php echo esc_html__('/ Night: ', 'varaus'); ?></small>
												<a class="theme-btn-s2" href="<?php echo esc_url(get_permalink()); ?>">
													<?php echo esc_html($read_more_txt); ?>
												</a>
											</div>
										</div>
									</div>
								<?php endwhile;
								wp_reset_postdata(); ?>
							</div>
						</div>
					</div>
				</div>
			</div>
<?php endif;
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Case widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Varaus_Case());
