<?php
/*
 * Elementor Varaus Gallery Widget
 * Author & Copyright: wpoceans
*/

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

class Varaus_Gallery extends Widget_Base
{

	/**
	 * Retrieve the widget name.
	 */
	public function get_name()
	{
		return 'tmx-varaus_gallery';
	}

	/**
	 * Retrieve the widget title.
	 */
	public function get_title()
	{
		return esc_html__('Gallery', 'varaus-core');
	}

	/**
	 * Retrieve the widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-flip-box';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 */
	public function get_categories()
	{
		return ['wpoceans-category'];
	}

	/**
	 * Retrieve the list of scripts the Varaus Gallery widget depended on.
	 * Used to set scripts dependencies required to run the widget.
	 */
	public function get_script_depends()
	{
		return ['tmx-varaus_gallery'];
	}

	/**
	 * Register Varaus Gallery widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'section_gallery',
			[
				'label' => esc_html__('Gallery Options', 'varaus-core'),
			]
		);
		$repeater = new Repeater();
		$repeater->add_control(
			'gallery_title',
			[
				'label' => esc_html__('Title Text', 'varaus-core'),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__('Title Text', 'varaus-core'),
				'placeholder' => esc_html__('Type title text here', 'varaus-core'),
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'gallery_image',
			[
				'label' => esc_html__('Gallery Image', 'varaus-core'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],

			]
		);
		$repeater->add_control(
			'gallery_link',
			[
				'label' => esc_html__('Link', 'varaus-core'),
				'type' => Controls_Manager::URL,
				'placeholder' => __('https://your-link.com', 'varaus-core'),
				'label_block' => true,
				'show_external' => true,
				'default' => [
					'url' => '',
					'is_external' => true,
					'nofollow' => true,
				],
			]
		);
		$this->add_control(
			'galleryLogos_groups',
			[
				'label' => esc_html__('Gallery Logos', 'varaus-core'),
				'type' => Controls_Manager::REPEATER,
				'default' => [
					[
						'gallery_title' => esc_html__('Gallery', 'varaus-core'),
					],

				],
				'fields' =>  $repeater->get_controls(),
				'title_field' => '{{{ gallery_title }}}',
			]
		);
		$this->end_controls_section(); // end: Section

		// Title
		$this->start_controls_section(
			'section_title_style',
			[
				'label' => esc_html__('Title', 'varaus-core'),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'varaus-core'),
				'name' => 'varaus_title_typography',
				'selector' => '{{WRAPPER}} .featured-area .featured-content a',
			]
		);
		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .featured-area .featured-content a' => 'color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_bg_color',
			[
				'label' => esc_html__('Background Color', 'varaus-core'),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .featured-area .featured-content a' => 'background-color: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'title_padding',
			[
				'label' => esc_html__('Title Padding', 'varaus-core'),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em'],
				'selectors' => [
					'{{WRAPPER}} .featured-area .featured-content a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section(); // end: Section
	}

	/**
	 * Render Gallery widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$galleryLogos_groups = !empty($settings['galleryLogos_groups']) ? $settings['galleryLogos_groups'] : [];

		// Turn output buffer on
		ob_start();
?>
		<div class="featured-area">
			<div class="grid">
				<div class="row">
					<?php 	// Group Param Output
					if (is_array($galleryLogos_groups) && !empty($galleryLogos_groups)) {
						foreach ($galleryLogos_groups as $each_logo) {

							$gallery_title = !empty($each_logo['gallery_title']) ? $each_logo['gallery_title'] : '';
							$image_url = wp_get_attachment_url($each_logo['gallery_image']['id']);
							$image_alt = get_post_meta($each_logo['gallery_image']['id'], '_wp_attachment_image_alt', true);

							$image_link = !empty($each_logo['gallery_link']['url']) ? $each_logo['gallery_link']['url'] : '';
							$image_link_external = !empty($each_logo['gallery_link']['is_external']) ? 'target="_blank"' : '';
							$image_link_nofollow = !empty($each_logo['gallery_link']['nofollow']) ? 'rel="nofollow"' : '';
							$image_link_attr = !empty($image_link) ?  $image_link_external . ' ' . $image_link_nofollow : '';

							$button = $image_link ? '<a href="' . esc_url($image_link) . '" ' . esc_attr($image_link_attr) . ' class="theme-btn" >' . esc_html($gallery_title) . '</a>' : '';
					?>
							<div class="col-lg-4 items col-md-6 col-sm-6 col-12">
								<div class="featured-wrap">
									<div class="featured-img">
										<img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>">
										<div class="featured-content">
											<?php echo $button; ?>
										</div>
									</div>
								</div>
							</div>
					<?php }
					} ?>
				</div>
			</div>
		</div>
<?php
		// Return outbut buffer
		echo ob_get_clean();
	}
	/**
	 * Render Gallery widget output in the editor.
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 */

	//protected function _content_template(){}

}
Plugin::instance()->widgets_manager->register(new Varaus_Gallery());
